/* 
   hotswap - Supports hotswapping of IDE devices.

   Copyright (C) 2001-2002 Tim Stadelmann

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*/



/** HEADERS **/

/* autoconf header */
#if HAVE_CONFIG_H
#  include <config.h>
#endif

/* ISO C headers */
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <locale.h>

/* POSIX headers */
#include <fcntl.h>
#include <mntent.h>

/* GNU headers */
#include <getopt.h>

/* system specific headers */
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <linux/hdreg.h>
#include <linux/cdrom.h>

/* program headers */
#include "hotswap.h"
#ifdef HAVE_LIBXML2
#  include "rcfile.h"
#endif

#ifdef ENABLE_NLS
#  include <libintl.h>
#  define _(Text) gettext (Text)
#else
#  define _(Text) Text
#endif



/** MACROS **/

/* the number of the default IDE controller */
#ifndef IDE_CONTROLLER
#define IDE_CONTROLLER 1
#endif /* IDE_CONTROLLER */

/* for convenience */
#define max(a,b) (((a) > (b)) ? (a) : (b))

/* the maximal number of significant bytes in the user's response to
   interactive yes/no questions */
#define RESPONSE_LENGTH 8



/** PROTOTYPES **/

/* Declare prototypes for the functions in this file.  See the
   individual functions for explanations.  */
static void usage (int status);
static int decode_switches (int argc, char **argv);
static int probe_ide ();
static int rescan_ide ();
static int unregister_ide ();
static int check_mounted_ide ();



/** GLOBAL VARIABLES **/

settings_t settings = {
  IDE_CONTROLLER,		/* controller */
  0,				/* nocheck */
  0				/* verbose */
};

char *program_name;		/* the name of the executable */
static char ide_device_model[40]; /* The model string from the IDE
				   device.  The maximum length of this
				   string is fixed, see
				   linux/hdreg.h.  */
static int ide_device_present;	/* whether an IDE device is present */
static const char *device_name;	/* the name of the device node */

/* default device names */
static const char* default_names[] = {
  "/dev/hda",
  "/dev/hdc",
  "/dev/hde",
  "/dev/hdg",
  "/dev/hdi",
  "/dev/hdk",
  "/dev/hdm",
  "/dev/hdo",
  "/dev/hdq",
  "/dev/hds"
};

/* Option flags and variables */
static struct option const long_options[] = {
  {"ide-controller", required_argument, 0, 'c'},
  {"help", no_argument, 0, 'h'},
  {"nocheck", no_argument, 0, 'n'},
  {"verbose", no_argument, 0, 'v'},
  {"version", no_argument, 0, 'V'},
  {NULL, 0, NULL, 0}
};



/** MAIN **/

int
main (int argc, char **argv)
{
  int i;
  int error;
  char response[RESPONSE_LENGTH];

  program_name = argv[0];

  /* Get the locale from the environment.  */
  setlocale (LC_ALL, "");

  /* Initialize gettext.  */
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  i = decode_switches (argc, argv);

#ifdef HAVE_LIBXML2
  /* Read the configuration file.  */
  init_rcfile ();
#endif /* not HAVE_LIBXML2 */  

  /* Get the device name corresponding to the fist device on the given
     IDE controller.  */
  device_name = default_names[settings.controller];

  /* Check whether a specific action was requested on the command
     line.  */

  if (i == argc)
    {
      /* No ACTION argument is present.  */

      /* Enter interactive mode.  */

      printf (_("\
hotswap %s\n\
Copyright 2001 Tim Stadelmann\n\
This program is free software, licensed under the conditions of the\n\
GNU General Public License version 2, or (at your option), any later\n\
version.\n\n\
"), VERSION);

      /* Check for an IDE device.  */
      error = probe_ide ();
      if (error)
	{
	  exit (EXIT_FAILURE);
	}

      /* This flag is set by probe_ide.  */
      if (ide_device_present)
	{
	  printf (_("\n\
The following IDE device is currently configured:\n\
%s\n\
Do you want to remove this device? "), ide_device_model);

	  fgets (response, RESPONSE_LENGTH, stdin);
	  if (rpmatch (response) != 1)
	    {
	      puts (_("Aborting"));
	      exit (EXIT_FAILURE);
	    }

	  /* Is the IDE device mounted?  */
	  error = check_mounted_ide ();
	  if (error != 0)
	    exit (EXIT_FAILURE);

	  /* Unregister the IDE device.  */
	  error = unregister_ide ();
	  if (error)
	    exit (EXIT_FAILURE);

	  puts (_("You can now remove the device from the module bay."));
	}
      else			/* No IDE device is present.  */
	{
	  puts (_("\
There is currently no IDE device configured.  (Floppy disk drives,\n\
batteries, and 'travel modules' are not managed by this utility.  If\n\
you want to swap such a module, you should remove it now.)\
"));
	}

      printf (_("\n\
Would you like to insert an IDE device into the module bay? "));

      fgets (response, RESPONSE_LENGTH, stdin);
      if (rpmatch (response) != 1)
	{
	  puts (_("Aborting"));
	  exit (EXIT_FAILURE);
	}

      printf (_("\
Please insert the new device into the module bay and press RETURN.\n"));
      getc (stdin);

      /* Rescan the IDE bus.  */
    rescan:
      error = rescan_ide ();
      if (error)
	exit (EXIT_FAILURE);

      if (ide_device_present)
	{
	  printf (_("\
The following IDE device has been configured successfully:\n\
%s\n"), ide_device_model);
	}
      else			/* No IDE device has been found.  */
	{
	  puts (_("\
No IDE device has been found in the module bay.  If you are sure that\n\
the device you want to configure connects to the IDE bus, make sure\n\
that the module is inserted completely.\n\
"));

	  printf (_("Do you want to try again? "));

	  fgets (response, RESPONSE_LENGTH, stdin);
	  if (rpmatch (response) != 1)
	    {
	      puts (_("Aborting"));
	      exit (EXIT_FAILURE);
	    }
	  else
	    {
	      /* Go back and attempt to rescan the IDE bus again.  */
	      goto rescan;
	    }
	}

      /* Return successfully.  */
      exit (EXIT_SUCCESS);
    }

  else if (i == argc - 1)
    {
      /* There is an ACTION argument.  */

      if (strcoll ("probe-ide", argv[i]) == 0)
	{
	  /* Detect the IDE device.  */
	  error = probe_ide ();
	  if (error)
	    {
	      exit (EXIT_FAILURE);
	    }
	  else
	    {
	      printf ("%s\n", ide_device_model);
	      exit (EXIT_SUCCESS);
	    }
	}

      else if (strcoll ("mounted-ide", argv[i]) == 0)
	{
	  /* Is the IDE device mounted? */
	  error = check_mounted_ide ();
	  if (error == -1)
	    exit (EXIT_FAILURE);

	  if (error == 1)
	    puts (_("yes"));
	  else
	    puts (_("no"));

	  exit (EXIT_SUCCESS);
	}

      else if (strcoll ("unregister-ide", argv[i]) == 0)
	{
	  if (!settings.nocheck)
	    {
	      /* Is the IDE device mounted? */
	      error = check_mounted_ide ();
	      if (error == -1)
		exit (EXIT_FAILURE);

	      if (error == 1)
		{
		  printf ("%s: ", program_name);
		  puts (_("\
at least one filesystem on the IDE device is currently mounted and\n\
cannot be removed\
"));
		  exit (EXIT_FAILURE);
		}
	    }

	  /* Unregister the IDE device.  */
	  error = unregister_ide ();
	  if (error)
	    exit (EXIT_FAILURE);
	  else
	    exit (EXIT_SUCCESS);
	}

      else if (strcoll ("rescan-ide", argv[i]) == 0)
	{
	  if (!settings.nocheck)
	    {
	      /* Is an IDE device already configured? */
	      error = probe_ide ();
	      if (error)
		exit (EXIT_FAILURE);
	      if (ide_device_present)
		{
		  printf ("%s: ", program_name);
		  printf (_("\
an IDE device is already configured:\n\
%s\n\
"), ide_device_model);
		  exit (EXIT_FAILURE);
		}
	    }

	  /* Rescan the IDE bus.  */
	  error = rescan_ide ();
	  if (error)
	    exit (EXIT_FAILURE);
	  else
	    exit (EXIT_SUCCESS);
	}

      else
	{
	  printf ("%s: ", program_name);
	  printf (_("%s: unknown action\n\n"), argv[i]);
	  usage (EXIT_FAILURE);
	}
    }

  else				/* Too many arguments are given.  */
    {
      printf ("%s: ", program_name);
      puts (_("too many arguments\n"));
      usage (EXIT_FAILURE);
    }

}



/** FUNCTIONS **/

/* Set all the option flags according to the switches specified.
   Return the index of the first non-option argument.  */
static int
decode_switches (int argc, char **argv)
{
  int c;			/* the option character */

  while ((c = getopt_long (argc, argv,
			   "c:"	/* controller */
			   "h"	/* help */
			   "n"	/* don't check */
			   "v"	/* verbose */
			   "V",	/* version */
			   long_options, (int *) 0)) != EOF)
    {
      switch (c)
	{
	case 'c':
	  /* Convert the argument into a number identifying the IDE
	     bus to be used.  If the conversion fails, or the value
	     lies outside the somewhat arbitrary allowable range,
	     return an appropriate error message.  */
	  errno = 0;
	  settings.controller = strtol (optarg, NULL, 10);
	  if (errno || settings.controller < 0 || settings.controller > 9)
	    {
	      fprintf (stderr, "%s: ", program_name);
	      fputs (_("invalid controller number\n"), stderr);
	      exit (EXIT_FAILURE);
	    }
	  break;
	case 'h':
	  usage (EXIT_SUCCESS);
	case 'n':
	  settings.nocheck = 1;
	  break;
	case 'v':
	  settings.verbose = 1;
	  break;
	case 'V':
	  printf (_("\
hotswap %s\n\
Copyright 2001 Tim Stadelmann\n\
This program is free software, licensed under the conditions of the\n\
GNU General Public License version 2, or (at your option), any later\n\
version.\n\
"), VERSION);
	  exit (EXIT_SUCCESS);
	default:
	  usage (EXIT_FAILURE);
	}
    }

  return optind;
}


static void
usage (int status)
{
  printf (_("%s - \
Supports hotswapping IDE devices.\n"), program_name);
  printf (_("Usage: %s [OPTION]... [ACTION]\n"), program_name);
  puts (_("\
\n\
Options:\n\
  -c, --ide-controller N     use IDE controller N\n\
  -h, --help                 print this help text and exit successfully\n\
  -n, --nocheck              do not perform sanity checks before\n\
                             attempting an action; use with care, as\n\
                             this option may cause actions to fail\n\
                             silently\n\
  -v, --verbose              print more information\n\
  -V, --version              print version information and exit successfully\n\
\n\
If no ACTION is specified, interactive mode is entered.  In this mode\n\
the program guides through the hotswapping process as appropriate for\n\
the current configuration.\n\
\n\
ACTION can be one of:\n\
  probe-ide                  detect the configured IDE device\n\
  mounted-ide                check whether the IDE device is mounted\n\
  unregister-ide             unregister the IDE device\n\
  rescan-ide                 scan the IDE bus for a new device\n\
\n\
You should note that Linux kernel requires hotswappable devices to be\n\
the only device on their IDE bus.\n\
\n\
Report bugs to <t.stadelmann1@physics.ox.ac.uk>.\
"));
  exit (status);
}


/* Detect whether the device is configured and obtain the model
   identification string.  Return -1 if an error has occured, zero
   otherwise.  The model is written into the global buffer
   ide_device_model; defaults for absent and unidentified devices are
   provided.  The global variable ide_device_present indicates whether
   a configured IDE device is present.  */

static int
probe_ide ()
{
  struct hd_driveid identity;
  int device;
  int error;

  if (settings.verbose)
    {
      printf ("%s: ", program_name);
      puts (_("probing the configured IDE device"));
    }

  ide_device_present = 0;

  /* Open the device.  O_NONBLOCK prevents the open from failing if no
     medium is inserted.  */
  device = open (device_name, O_RDONLY | O_NONBLOCK);
  if (device < 0)
    {
      /* A failure generally means that no drive is present.  */
      strncpy (ide_device_model, _("absent"), 40);
      /* This is not an error condition from the point of view of this
         program.  */
      return 0;
    }

  /* Get the drive identification.  */
  error = ioctl (device, HDIO_GET_IDENTITY, &identity);

  if ((errno == ENOMSG) || (!error && (identity.model[0] == '\0')))
    {
      /* Deal with IDE devices that do not implement drive
         identification.  Just in case, I've never actually seen one.  */
      strncpy (ide_device_model, _("unidentified IDE device"), 40);
      error = 0;
      ide_device_present = 1;
    }
  else if (error)
    {
      fprintf (stderr, "%s: ", program_name);
      perror (_("obtaining IDE drive indentification failed"));
    }
  else
    {
      strncpy (ide_device_model, identity.model, 40);
      ide_device_present = 1;
    }

  /* Close the device.  */
  close (device);

  return error;
}


/* Rescan the secondary IDE bus.  Returns -1 if an error has occured,
   zero otherwise.  */
static int
rescan_ide ()
{
  int device;
  /* These identify the second IDE bus.  See linux/ide.h.  */
  int args[] = { 0x170, 0x376, 15 };
  int error;

  if (settings.verbose)
    {
      fprintf (stderr, "%s: ", program_name);
      puts (_("rescanning the IDE bus"));
    }

  /* The HDIO_SCAN_HWIF ioctl can be called on any IDE device.  We
     call it on /dev/hda since it is reasonable to assume that this
     device will always be available on a system using IDE.  */
  device = open ("/dev/hda", O_RDONLY);
  if (device < 0)
    {
      fprintf (stderr, "%s: ", program_name);
      perror (_("opening IDE device failed"));
      return -1;
    }

  /* Issue the ioctl.  */
  error = ioctl (device, HDIO_SCAN_HWIF, args);
  if (error != 0)
    {
      if (errno == EIO)		/* See ide.c in the kernel.  */
	{
	  /* No IDE device found.  */
	  error = 0;
	}
      else
	{
	  fprintf (stderr, "%s: ", program_name);
	  perror (_("rescanning the IDE bus failed"));
	}
    }

  /* Close the device.  */
  close (device);
  
  /* Check for an IDE device.  */
  
  if (!error)
    {
      error = probe_ide ();
    }

#ifdef HAVE_LIBXML2
  if (!error)
    {
      /* Run the post-install script.  */
      run_script (ide_device_model, HOTSWAP_POST_INSERT);
    }
#endif /* not HAVE_LIBXML2 */

  return error;
}


/* Unregister the device configured as MASTER on the secondary IDE
   bus.  Returns -1 if an error has occured, zero otherwise.  */
static int
unregister_ide ()
{
  int device;
  int error;

  if (settings.verbose)
    {
      fprintf (stderr, "%s: ", program_name);
      puts (_("unregistering the IDE device"));
    }
  
#ifdef HAVE_LIBXML2
  /* Run the pre-remove script.  */
  run_script (ide_device_model, HOTSWAP_PRE_REMOVE);
#endif /* not HAVE_LIBXML2 */

  /* If the device is a CDROM, it might still be playing.  Try and
     stop it in this case.  If it isn't a CDROM, this will generate an
     error which is ignored.  */

  /* Open the potential CDROM device; O_NONBLOCK prevents the open
     from failing if no medium is present. */
  device = open (device_name, O_RDONLY | O_NONBLOCK);
  if (device < 0)
    {
      fprintf (stderr, "%s: ", program_name);
      perror (_("opening IDE device failed"));
    }
  /* Issue the ioctl.  */
  error = ioctl (device, CDROMSTOP);
  /* Close the device.  */
  close (device);

  /* The HDIO_UNREGISTER_HWIF ioctl can be called on any IDE
     device.  */
  device = open ("/dev/hda", O_RDONLY);
  if (device < 0)
    {
      fprintf (stderr, "%s: ", program_name);
      perror (_("opening IDE device failed"));
      return -1;
    } 

 /* Issue the ioctl.  */
  error = ioctl (device, HDIO_UNREGISTER_HWIF, 1);
  if (error)
    {
      fprintf (stderr, "%s: ", program_name);
      perror (_("unregistering the IDE device failed"));
    }

  /* Close the device.  */
  close (device);

  /* Return on error.  */
  if (error)
    return error;

#ifdef HAVE_LIBXML2
  /* Run the post-remove script.  */
  run_script (ide_device_model, HOTSWAP_POST_REMOVE);
#endif /* not HAVE_LIBXML2 */

  /* The ioctl only returns an error if the calling process does not
     have root privileges.  Unregistering the device might still fail,
     for example if it is opened by some other process.  The only way
     to catch this is to probe for the device again.  Sigh.  */
  error = probe_ide ();
  if (error)
    {
      exit (EXIT_FAILURE);
    }

  if (ide_device_present)
    {
      fprintf (stderr, _("\
%s: unregistering the IDE device failed: device is probably busy\n\
"), program_name);
      error = -1;
    }

  return error;
}


/* Check whether the IDE device or a partition on it is mounted.
   Returns -1 for an error, 1 if at least one mount is active, and 0
   otherwise.  */
int
check_mounted_ide ()
{
  FILE *mtab;
  struct mntent *entry;
  struct stat device_info;
  struct stat fs_info;
  int mounted = 0;

  if (settings.verbose)
    {
      fprintf (stderr, "%s: ", program_name);
      puts (_("checking whether the IDE device is mounted"));
    }

  /* Get device identifier (major & minor code).  */
  if ( stat (device_name, &device_info) != 0 )
    {
      fprintf (stderr, "%s: stat %s failed ", program_name);
      perror (NULL);
      return -1;
    }
  
  
  /* Open /etc/mtab.  */
  mtab = setmntent ("/etc/mtab", "r");

  if (mtab == NULL)
    {
      fprintf (stderr, "%s: ", program_name);
      perror (_("opening /etc/mtab failed"));
      return -1;
    }

  /* Loop through the entries in /etc/mtab.  */
  for (;;)
    {
      entry = getmntent (mtab);
      if (entry == NULL)
	break;

      /* Compare the entry's device identifier and our device identifier.  */
      if ( !stat (entry->mnt_fsname, &fs_info) )
	{
	  if ( fs_info.st_rdev == device_info.st_rdev )
	    {
	      mounted = 1;
	      /* At least one file system is mounted.  */
	      break;
	    } 
	}
    }

  /* Close /etc/mtab.  */
  endmntent (mtab);

  return mounted;
}
